import { Request, Response } from 'express';
import { sketchesService } from './service';

function hasFeature(vendor: any, featureKey: string): boolean {
  return vendor?.features?.includes(featureKey);
}

export const sketchesController = {
  // POST /sketches (multipart)
  async upload(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'sketch_upload')) {
      return res.status(403).json({ error: 'Sketch upload not available on your plan.' });
    }
    if (!req.file) return res.status(400).json({ error: 'No file uploaded.' });
    const { data, error } = await sketchesService.uploadSketch(vendor, req.file, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // POST /sketches/:id/enhance
  async enhance(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'ai_sketch_enhance')) {
      return res.status(403).json({ error: 'AI sketch enhancement not available on your plan.' });
    }
    // Device guard for SVG (frontend should enforce, but double-check here)
    if (req.body.enhancement_type === 'svg' && req.device !== 'desktop' && req.device !== 'tablet') {
      return res.status(403).json({ error: 'SVG enhancement only allowed on desktop/tablet.' });
    }
    const { id } = req.params;
    const { data, error } = await sketchesService.enhanceSketch(vendor, id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /sketches
  async list(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });
    const { customer_id, invoice_id, tags } = req.query;
    const filters: any = { customer_id, invoice_id, tags };
    const { data, error } = await sketchesService.listSketches(vendor.id, filters);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /sketches/:id
  async get(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });
    const { id } = req.params;
    const { data, error } = await sketchesService.getSketch(vendor.id, id);
    if (error) return res.status(404).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // DELETE /sketches/:id
  async delete(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor) return res.status(401).json({ error: 'Unauthorized' });
    const { id } = req.params;
    const { data, error } = await sketchesService.deleteSketch(vendor.id, id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 